<?php
$metaTitle = "svgPHPGrapher Syntax Examples";
$metaDescription = "Syntax for svg graphs drawn using svgPHPGrapher, with examples of how to use the library.";
$katex=1;

include('svgPHPGrapher2018-12-18.php');

$brdNum=0; // For svg id's
?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,minimum-scale=1.0, maximum-scale=1.0" />
<title><?php echo $metaTitle; ?></title>
<meta name="description" content="<?php echo $metaDescription; ?>" />
<style type="text/css">
body{max-width:900px;font-family:Arial, Helvetica, sans-serif;font-size:0.95em}
.int{border:1px solid #5b8a9b;border-radius:0.4em;padding:0 0.5em 0.8em;
-webkit-touch-callout:none;-webkit-user-select:none;-khtml-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none }
.info{width:200px;font-size:0.9em;display:inline-block}
body{background:#eff;max-width:100%}
svg{overflow:hidden}
.svgphpgrapherwarn{color:#d00;font-weight:bold}
.svgphpWrap{position:relative;margin-left:10px}
.svgphpWrap a:hover{cursor:default}
.svgHtml{position:absolute;font-size:0.85em}
@font-face {font-family: MathJax_Math-italic;
    src: url('MathJax_Math-Italic.eot');
    src: url('MathJax_Math-Italic.eot?#iefix') format('embedded-opentype'),
			url('MathJax_Math-Italic.woff') format('woff');
	 }
@font-face {font-family: MathJax_Main;
    src: url('MathJax_Main-Regular.eot');
    src: url('MathJax_Main-Regular.eot?#iefix') format('embedded-opentype'),
			url('MathJax_Main-Regular.woff') format('woff') }
i,em{font-family:MathJax_Math-italic,"Times New Roman",Times,serif;font-size:1.2em;font-style:normal /* To stop double italic on MacOS */}			
.intmath{font-family:MathJax_Main,"Times New Roman",Times,serif;white-space:nowrap}
pre, code{font-size:1.1em}
pre{margin:0 0 0 1em;white-space: pre-wrap}
.longCode{padding:0;margin:0 0 0 1em}
.current{color:#888;font-size:0.85em}
code.indent{margin:0 0 0 1em}
</style>

</head>
<body>
<div id="wrapper">

<p class="bcrumb"><a href="/">Home</a> | <a href="/help/math-display-experiments.php">Math Display Experiments</a> | <?php echo $metaTitle; ?></p>
<small>Page by Murray Bourne, <a href="/">IntMath.com</a>. Last updated: <?php echo date("d M Y", filemtime(__FILE__)); ?></small>
<h1><?php echo $metaTitle; ?></h1>



<p><b>svgPHPGrapher</b> is a PHP-based mathematical grapher that produces svg images. It's a highly-modified PHP port of <a href="http://www1.chapman.edu/~jipsen/svg/asciisvg.html">ASCIIsvg.js</a> by Peter Jipsen.</p>
<p>The advantages of producing stand-alone svg output (as opposed to creating the graph with javascript, which is the case in ASCIIsvg and JSXGraph) is that it's quicker, because there is less for the browser to download (no javascript), and there is no javascript processing required during page load to produce the graph. It's also quicker on mobile devices, because the server does the processing work, rather than the device and its browser.</p>








<h2>svgPHPGrapher Example Chapters</h2>

<p>Apart from the examples below, you can see example graph output from svgPHPGrapher in these IntMath chapters:</p>
<ul>
  <li><a href="/inequalities/inequality.php">Inequalities</a></li>
  <li><a href="/equations-of-higher-degree/polynomial-equations.php">Polynomial Equations (Equations of Higher Degree</a></li>
  <li><a href="/exponential-logarithmic-functions/exponential-log-functions-intro.php">Exponential and Logarithmic Functions</a></li>
  <li><a href="/functions-and-graphs/functions-graphs-intro.php">Functions and Graphs</a></li>
  <li><a href="/quadratic-equations/quadratic-equations-intro.php">Quadratic Equations</a></li>
  <li><a href="/systems-of-equations/systems-equations.php">Systems of Equations</a></li> 
</ul>

<h2>On this page...</h2>
<ul>
  <li><a href="#brdreq">Board Requirements</a></li>
  <li><a href="#axes">Drawing a board with axes</a></li>  
  <li><a href="#plot">Plotting Mathematical Functions</a></li> 
  <li><a href="#line">Lines</a></li>  
  <li><a href="#seg">Segments and inequalities</a></li> 
  <li><a href="#log">Log-log &amp; Semi-log Graphs </a></li> 
  <li><a href="#polar">Polar Graphs</a></li> 
  <li><a href="#parametric">Parametric Curves</a></li> 
  <li><a href="#card">Cardinal Splines</a></li>  
  <li><a href="#geom">Geometric Shapes (circle, square, etc)</a></li>  
</ul>



<h2 id="brdreq">Board Requirements</h2>
<p>Somewhere at the top of your page, add the following (this grabs the latest version of svgPHPGrapher.php).</p>
<pre>&lt;?php
include('svgPHPGrapher2018-12-18.php');

$brdNum=0; // For svg id's
?&gt;
</pre>


<h2 id="axes">Drawing a Board with Axes</h2>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=''; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//

doOutput($svgArr,$htmlArr);
?>

<p>The board, grid and axes above were produced using the following code:</p>

<pre>
&lt;?php
// Create the board
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3; $yMax=''; 
$padding = 15;
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);

// Create the axes
$xAxisTicks = 1;	$yAxisTicks = 2;
$axisNumbering = "labels";
$verticalGridGap = 2;	$horizontalGridGap = 2;
$gridAttrs = 'stroke:#777; stroke-width:0.5';
$axesAttrs = 'stroke:#444; stroke-width:1';
$xAxisVbl = "x";	$yAxisVbl = "y";
$arrows = 1;
$svgArr[] = axes($xAxisTicks,$yAxisTicks, $axisNumbering, $verticalGridGap,$horizontalGridGap, $xAxisVbl,$yAxisVbl, $gridAttrs,$axesAttrs, $arrows, $brdParams);

// Echo out the results
doOutput($svgArr,$htmlArr);
?&gt;
</pre>

<h3>Explanation</h3>
<p>Each board on the page automatically gets a sequential ID:</p>
<pre>$boardID = "svgphp-$brdNum"; $brdNum++;</pre>
<p>This is your graph size (in pixels):</p>
<pre>$width = 310; $height = 200;</pre>
<p>This is the range of possible values in the horizontal and vertical axes.</p>
<pre>$xMin=-4; $xMax=6; $yMin=-3;  $yMax=''; </pre>
<p>Leaving <code>$yMax</code> as an empty string means the board will have equal-scaled axes.</p>
<p>This next one gives some space around the edge so you can see all of the axes numbers (a good default size is 15):</p>
<pre>$padding = 15;</pre>
<p>This is the first function call where you initialize the board. Nothing to do here, as you have defined all that's needed above.</p>
<p class="longCode"><code>list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);</code></p>
<p>Next is the function that echoes out the final output:</p>
<pre>doOutput($svgArr,$htmlArr);</pre>

<h3>Adding the axes</h3>
<p>The gaps between ticks on your axes:</p>
<pre>$xAxisTicks = 1;	$yAxisTicks = 2;</pre>
<p>If you want the numbers on your axes to show, put "labels" if so, or an empty string if not: "":</p>
<pre>$axisNumbering = "labels";</pre>
<p>The gaps between grids (<i>x</i>-direction, then <i>y</i>-direction):</p>
<pre>$verticalGridGap = 2;	$horizontalGridGap = 2;</pre>
<p>The <code>$gridAttrs</code> and <code>$axesAttrs</code> have defaults (see the top of the svgPHPGrapher function), but you can override colors and thickness if you want.</p>
<p>Here's where you state what axis variables you want. Leaving them blank (like <code>$xAxisVbl = ''</code>) will mean no axis label will appear.</p>
<pre>$xAxisVbl = "x";	$yAxisVbl = "y";</pre>
<p>This is if you want arrows on your axes (value 1) or not (value 0).
<pre>$arrows = 1;</pre>
<p>Now for the axes function. (If you put <code>$yAxisTicks = ''</code> it means no y-axis.)</p>
<p class="longCode"><code>$svgArr[] = axes($xAxisTicks,$yAxisTicks, $axisNumbering, $verticalGridGap,$horizontalGridGap, $xAxisVbl,$yAxisVbl, $gridAttrs,$axesAttrs, $arrows, $brdParams); </code></p>
<p>This closes the svg tag and prints out the resulting graph to the screen:</p>
<pre>$svgArr[] = '&lt;/svg>';
doOutput($svgArr,$htmlArr);
</pre>

<h3>Multiples of `pi` on axes</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-8; $xMax=10; $yMin=-4;  $yMax=''; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(pi(),1,"pi,numbers",pi(),2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//

doOutput($svgArr,$htmlArr);
?>

<p>The above axes use the following call to the <code>axes</code> function:</p>

<pre>$svgArr[] = axes(pi(),1,"pi,numbers",pi(),2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);</pre>


<h2 id="plot">Plotting Mathematical Functions</h2>

<?php

$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-0.2; $xMax=1.5; $yMin=-0.05;  $yMax=0.2; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(0.2,0.05,"labels",1,0.05, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("x^4 + 0.4x^3 - 6.49x^2 + 7.244x - 2.112",0.5,$xMax,80, $attrs,$brdParams);
$attrs = 'stroke:none; fill:#f0f';
$svgArr[] = dot([1.2,0],3,$attrs, $brdParams); //dot($center, $radius, $attrs, $brdParams)
$svgArr[] = dot([0.5,0],3,$attrs, $brdParams);
$svgArr[] = dot([1.1,0],3,$attrs, $brdParams);
//

doOutput($svgArr,$htmlArr);
?>

<p>The above polynomial was plotted as follows.</p>
<p>The function is declared like this:</p>
<pre>$fun = "x^4 + 0.4x^3 - 6.49x^2 + 7.244x - 2.112";</pre>
<p>You indicate the start and end <i>x</i>-values for the curve as follows (leaving <code>$xStart</code> blank means the curve will start from the far left side of the graph, and leaving <code>$xEnd</code> blank means the domain extends all the way to the right):</p>
<pre>$xStart = 0.5;	$xEnd = '';</pre>
<p>You can set the number of points for the curve (lower numbers means  a "chunkier" graph, higher means a smoother graph but larger file size):</p>
<pre>$numPts = 80;</pre>
<p>These are the attributes for the curve:</p>
<pre>$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';</pre>
<p>The syntax for the <code>plot</code> function is:</p>
<pre>$svgArr[] = plot($fun, $xStart,$xEnd, $numPts, $attrs,$brdParams);</pre>
<p>The following syntax produces the dots at the <i>x</i>-axis intersections:</p>
<pre>$attrs = 'stroke:none; fill:#f0f';</pre>
<pre>$svgArr[] = dot([1.2,0], 3, $attrs,$brdParams);</pre>
<pre>$svgArr[] = dot([0.5,0], 3, $attrs,$brdParams);</pre>
<pre>$svgArr[] = dot([1.1,0], 3, $attrs,$brdParams);</pre>
<p>The parameters for the dot function are:</p>
<pre>$center, $radius, $attrs, $brdParams</pre>

<p><b>NOTE:</b> Possible independent variables are `x` and `t`.</p>

<h2>Vertical Graph Cases</h2>
<p>Here's a common problem when constructing graphs using a computer-based grapher. If the graph becomes very steep, the only way to ensure all of it is included is to have a very large number of sample points. But this is not a good idea as:</p>
<ul><li>It results in a large file size</li>
<li>It's not necessary to have a large number of points for the graph in parts where it is not so steep.</li>
</ul>
<p>Let's illustrate the problem with an example. Here I'm trying to draw the graph of `x=y^2+2`. This is a relation (not a function) and I need to draw it in 2 halves:</p>
<ol><li>The top half, `y=sqrt(x-2)`; and</li>
<li>The bottom half, `y=-sqrt(x-2)`.</li>
</ol>
<p>As you can see, there is a gap near `x=2` where the top and bottom half should meet.</p> 

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-3; $xMax=16; $yMin=-3.5;  $yMax=4; 
$padding=15;
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
$svgArr[] = axes(5,1,"labels",0,0, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; fill:none; stroke-width:1.5';
// Empty $x_min,$x_max means full width graph
//$svgArr[] = plot("log(x+1)",'','',100, $attrs,$brdParams);   //plot($fun,$x_min,$x_max,$points,$id, $attrs,$brdParams)
$svgArr[] = plot("sqrt(x-2)",'','',60, $attrs,$brdParams);
$svgArr[] = plot("-sqrt(x-2)",'','',60, $attrs,$brdParams); 
//
doOutput($svgArr,$htmlArr);
?>

<p>Here's the syntax I've used:</p>
<pre>
$svgArr[] = plot("sqrt(x-2)",'','',60, $attrs,$brdParams);
$svgArr[] = plot("-sqrt(x-2)",'','',60, $attrs,$brdParams);
</pre>
<p>In svgPHPGrapher, there is a built-in attempt to solve this. When the graph is steep, the number of points sampled increases by a factor of 10. But even with this enhancement, you can see there still is a gap.</p>
<p>If I increase the number of points for each of the arms of the graph from 60 to 400 (giving a total of 800 points for the 2 arms plus the extra ones generated near the gap, I still get a gap.</p>
<p>Here's one way to solve this issue. We graph the inverse instead.</p>
<h3>Graph the Inverse</h3>
<p>It's possible in svgPHPGrapher to plot an inverse. In this example, the inverse of `y=+-sqrt(x-2)` is `x=y^2+2`. Here's what the graph looks like.</p>


<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-3; $xMax=16; $yMin=-3.5;  $yMax=4; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(5,1,"labels",0,0, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("y^2+2",'','',40, $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>This is the syntax used this time:</p>

<pre>$svgArr[] = plot("y^2+2",'','',40, $attrs,$brdParams);</pre>

<p>Notice the use of the `y` variable. Internally, the grapher takes `y`-values from the minimum `y`-value to the maximum `y`-value. You'll notice that with only 40 sample points, the curve is smooth and connected, and there is no trouble when the graph is vertical at `x=2`.</p>


<h3>Graphs with Discontinuities</h3>
<p>Similar problems occur when we try to draw periodic graphs with vertical asymptotes, for example, `y=tan x`.</p>
<p>You can see that the grapher has connected the top of each positive arm of the curve to the bottom of the negative (supposedly infinite) negative next arm. You can also see the graph doesn't go down all the way to the minimum negative value.</p>
<?php

$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-15;  $yMax=15; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(pi(),5,"pi,numbers",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("tan(x)",'','',70, $attrs,$brdParams);
$markers = '';
/*
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none; stroke-dasharray:5,5';
for($i=-1; $i<5; $i+=2) {
	$svgArr[] = line([$i*pi()/2, $yMin], [$i*pi()/2, $yMax], $attrs,$brdParams);
}*/
//

doOutput($svgArr,$htmlArr);
?>

<p>Using the inverse function, we can get a better looking tan curve, with no joins and a small number of points.</p>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=7.5; $yMin=-15;  $yMax=15; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(pi(),5,"pi,numbers",0,0, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
for($k=-pi();$k<=2*pi();$k+=pi()) {
	$attrs = 'stroke:#165a71; stroke-width:1.5;fill:none';	
	$svgArr[] = plot("atan(y)+$k",$yMin,$yMax,45, $attrs,$brdParams);
	$attrs = 'stroke:#165a71; stroke-width:1; stroke-dasharray:5,5';	
	$svgArr[] = line([$k+pi()/2, $yMin], [$k+pi()/2, $yMax], $attrs,$brdParams);
}
//
doOutput($svgArr,$htmlArr);
?>

<p>What I've done here is to graph the inverse tan curve using the `y` variable: `"atan"(y)` (which means `arctan (y)`). I then replicate that single curve as many times as needed to cover the board.
<p>Here's the syntax for the loop:</p>

<p class="longCode">
<code>for($k=-pi(); $k&lt;=2*pi(); $k+=pi()) {</code><br />
<code class="indent">  $attrs = 'stroke:#165a71; stroke-width:1.5;fill:none';</code><br />	
<code class="indent">  $svgArr[] = plot("atan(y)+$k", $yMin,$yMax, 60, $attrs,$brdParams);</code><br /><br />
<code class="indent">  $attrs = 'stroke:#165a71; stroke-width:1; stroke-dasharray:5,5';</code><br />	
<code class="indent">  $svgArr[] = line([$k+pi()/2, $yMin], [$k+pi()/2, $yMax], $attrs,$brdParams);</code><br />
<code>}</code>


<p>The first 2 lines in the loop are responsible for repeating the single `"atan"(y)` curve, and the last 2 lines produce the dashed vertical asymptotes.</p>

<p>Here are the cuves `y="atan"x` (in magenta) and `x="atan"y` (in green) drawn on the one set of axes. Since they are inverses, they are reflections in the line `y=x`, which is also shown.</p>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 310;
$xMin=-6; $xMax=6; $yMin=-6;  $yMax=''; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';	
$svgArr[] = plot("atan(y)",$yMin,$yMax,30, $attrs,$brdParams);
$attrs = 'stroke:#f0f; stroke-width:1.5; fill:none';
$svgArr[] = plot("atan(x)",$xMin,$xMax,30, $attrs,$brdParams);
$attrs = 'stroke:#165a71; stroke-width:1; fill:none; stroke-dasharray:5,5';	
$svgArr[] = line([$xMin,$xMin], [$xMax,$xMax], $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>Here's another example where I've plotted a curve `y=sin x` and its inverse `x= sin y` on the one set of axes, and shown the reflection line `y=x`.</p>


<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 310;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=''; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("sin(y)",$yMin,$yMax,40, $attrs,$brdParams);
$attrs = 'stroke:#f0f; stroke-width:1.5; fill:none';
$svgArr[] = plot("sin(x)",'','',40, $attrs,$brdParams);
$attrs = 'stroke:#165a71; stroke-width:1; fill:none; stroke-dasharray:5,5';	
$svgArr[] = line([$xMin,$xMin], [$xMax,$xMax], $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>


<p>Here's another place where I needed to use the inverse function idea.</p>

<?php

$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 220;
$xMin=-5; $xMax=5; $yMin=-3;  $yMax=3; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";



// $dy = '' or 0 means no y-axis
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("sqrt((y^2 + 4)/3)",$yMin,$yMax,40, $attrs,$brdParams);
$svgArr[] = plot("-sqrt((y^2 + 4)/3)",$yMin,$yMax,40, $attrs,$brdParams);
$attrs = 'stroke:#f0f; stroke-width:1.5; fill:none';
$svgArr[] = ellipse([0,0], sqrt(10), sqrt(2.5), '', $attrs,$brdParams);
$attrs = 'stroke:none; fill:#f0f';
$svgArr[] = dot([sqrt(2),sqrt(2)],3,$attrs, $brdParams); //dot($center, $radius, $attrs, $brdParams)
$svgArr[] = dot([sqrt(2),-sqrt(2)],3,$attrs, $brdParams);
$svgArr[] = dot([-sqrt(2),sqrt(2)],3,$attrs, $brdParams);
$svgArr[] = dot([-sqrt(2),-sqrt(2)],3,$attrs, $brdParams);
//

doOutput($svgArr,$htmlArr);
?>

<p>The 2 arms of the green hyperbola would have proved difficult using ordinary `f(x)` approach. This is the syntax used:</p>

<pre>
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("sqrt((y^2 + 4)/3)",$yMin,$yMax,40, $attrs,$brdParams);
$svgArr[] = plot("-sqrt((y^2 + 4)/3)",$yMin,$yMax,40, $attrs,$brdParams);
</pre>


<p>Finally, some graphs have both a steep part and a gently rising part. For example, the graph of `y=log_10 x`. Here I've drawn it in 2 parts.</p>

<?php
// /exponential-logarithmic-functions/2-graphs-exp-log-fns.php Example 2 answer (2 parts...)
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 250;
$xMin=-0.5; $xMax=4.5; $yMin=-2.5;  $yMax=1.5; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$svgArr[] = axes(1,1,"labels",1,1, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("(ln(x))/(ln(10))",0.1,'',40, $attrs,$brdParams);
$attrs = 'stroke:#f0f; stroke-width:1.5; fill:none';
$svgArr[] = plot("10^y",$yMin,-1,40, $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>The green part (quite shallow, above `y=-1`) is:</p>
<pre>$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("(ln(x))/(ln(10))",0.1,'',40, $attrs,$brdParams);
</pre>
<p>The magenta part (almost vertical to start with, below `y=-1`) is:</p> 
<pre>$attrs = 'stroke:#f0f; stroke-width:1.5; fill:none';
$svgArr[] = plot("10^y",$yMin,-1,40, $attrs,$brdParams);</pre>


<h2 id="line">Line through 2 points</h2>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=3; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
// 
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5';
$svgArr[] = line([-2,-1], [2,2], $attrs,$brdParams);
$attrs = 'stroke:none; fill:#165a71';	
$svgArr[] = dot([-2,-1],3,$attrs, $brdParams);
$svgArr[] = dot([2,2],3,$attrs, $brdParams);
//
doOutput($svgArr,$htmlArr);
?>
<p>A <b>line</b> extends across the whole board. Here is the syntax for the above line, which passes through `(-2,-1)` and `(2,2)`:</p>
<pre>
$attrs = 'stroke:#165a71; stroke-width:1.5';
$svgArr[] = line([-2,-1], [2,2], $attrs,$brdParams);
</pre>


<h2 id="seg">Segments</h2>


<h3>Simple segment</h3>


<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=3; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
// 
$xAxisVbl = "x";	$yAxisVbl = "y";
$svgArr[] = axes(1,1,"labels",1,1, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$p = [-3,2];  $q = [3,-3];  
$attrs = 'stroke:#165a71; stroke-width:2;';
$svgArr[] = segment($p, $q,  $attrs,$brdParams);
$attrs = 'stroke:none; fill:#165a71';	
$svgArr[] = dot([-3,2],3,$attrs, $brdParams);
$svgArr[] = dot([3,-3],3,$attrs, $brdParams);
//

doOutput($svgArr,$htmlArr);
?>

<p>Here is the syntax for the segment joining `(-3,2)` and `(3,-3)`:</p>

<pre>$p = [-3,2];  $q = [3,-3];  
$attrs = 'stroke:#165a71; stroke-width:2;';
$svgArr[] = segment($p, $q,  $attrs,$brdParams);
</pre>

<p>Segments can be embellished with various markers (dots or arrows).</p>

<h3>Segment-Arrow</h3>
<p>This one is useful for drawing vectors. You can change the color and size of the arrow head.</p>


<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=1; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
// 
$xAxisVbl = "x";	$yAxisVbl = "y";
$svgArr[] = axes(1,1,"labels",1,1, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$p = [1,-1];  $q = [1,-2];  
$segAttrs = 'stroke:#165a71; stroke-width:2;';
$arrAttrs = 'triWidth:8; triHeight:12; stroke:#165a71; stroke-width:1; fill:#165a71';
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
$p = [2,-2.5];  $q = [5,-1];
$segAttrs = 'stroke:#888; stroke-width:2;'; 
$arrAttrs = 'triWidth:15; triHeight:20; stroke:#165a71; stroke-width:1; fill:#4f4;';
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>
<p>Here is the syntax used for the first segment, pointing down:</p>
<pre>
$p = [1,-1];  $q = [1,-2];  
$segAttrs = 'stroke:#f0f; stroke-width:2;';
$arrAttrs = 'triWidth:8; triHeight:12; stroke:#165a71; stroke-width:1; fill:#3df;';
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);</pre>
<p>Here is the second grey segment, with green arrow head:</p>
<pre>$p = [2,-2.5];  $q = [5,-1];
$segAttrs = 'stroke:#888; stroke-width:2;'; 
$arrAttrs = 'triWidth:15; triHeight:20; stroke:#165a71; stroke-width:1; fill:#4f4;';
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
</pre>


<h3>Dot-Segment-Dot</h3>
<p>You can have different-size (and colored) dots on the end of your segments, as follows:</p>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=3; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
// 
$xAxisVbl = "x";	$yAxisVbl = "y";
$arrows = 1;
$svgArr[] = axes(1,1,"labels",1,1, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$p = [-3,0];  $q = [4,2];  
$dot0Attrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$segAttrs = 'stroke:#165a71; stroke-width:2;';
$dot1Attrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
$p = [-3,-3];  $q = [4,-1];  
$dot0Attrs = 'radius:10; stroke:#165a71; stroke-width:2; fill:#f0f; fill-opacity:0.4';
$segAttrs = 'stroke:#888; stroke-width:2;';
$dot1Attrs = 'radius:3; stroke:none; fill:#f55';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
//
doOutput($svgArr,$htmlArr);
?>
<p>Here is the syntax used for the first "dot-segment-dot" segment. Note the 2 dots ("dot0" and "dot1" have their own attributes.)</p>
<pre>
$p = [-3,0];  $q = [4,2];  
$dot0Attrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$segAttrs = 'stroke:#165a71; stroke-width:2';
$dot1Attrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
</pre>
<p>Here is the second multi-colored and multi-sized one:</p>
<pre>
$p = [-3,-3];  $q = [4,-1];  
$dot0Attrs = 'radius:10; stroke:#165a71; stroke-width:2; fill:#f0f; fill-opacity:0.4';
$segAttrs = 'stroke:#888; stroke-width:2;;
$dot1Attrs = 'radius:3; stroke:none; fill:#f55';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
</pre>



<h3>Dot-Segment-Arrow</h3>
<p>You can have your segment starting with a dot, and ending with an arrow, as follows:</p>




<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=1; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
// 
$xAxisVbl = "x";	$yAxisVbl = "y";
$arrows = 1;
$svgArr[] = axes(1,1,"labels",1,1, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$dotAttrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$segAttrs = 'stroke:#165a71; stroke-width:2;';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:none';
$svgArr[] = dotSegArrow([-2,-2], [2,-3], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
$svgArr[] = dotSegArrow([-2,-1], [0,0], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
$dotAttrs = 'radius:15; stroke:#165a71; stroke-width:2; fill:#165a71;opacity:0.5';
$segAttrs = 'stroke:#165a71; stroke-width:4;opacity:0.5';
$arrAttrs = 'triWidth:20; triHeight:20; stroke:#165a71; stroke-width:1; fill:#165a71;opacity:0.5';
$svgArr[] = dotSegArrow([5,1], [3,-1], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
$dotAttrs = 'radius:3; stroke:#165a71; stroke-width:2; fill:#165a71';
$segAttrs = 'stroke:#165a71; stroke-width:1';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:#165a71';
$svgArr[] = dotSegArrow([5,-2.5], [2,-3], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>The first 2 segments (on the left) share the same attributes:</p>

<pre>$dotAttrs = 'radius:5; stroke:#165a71; stroke-width:2; fill:none';
$segAttrs = 'stroke:#165a71; stroke-width:2;';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:none';
$svgArr[] = dotSegArrow([-2,-2], [2,-3], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
$svgArr[] = dotSegArrow([-2,-1], [0,0], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);</pre>

<p>This is the segment with larger features at top right (I've toned it down by reducing the opacity):</p>
<pre>
$dotAttrs = 'radius:15; stroke:#165a71; stroke-width:2; fill:#165a71;opacity:0.5';
$segAttrs = 'stroke:#165a71; stroke-width:4;opacity:0.5';
$arrAttrs = 'triWidth:20; triHeight:20; stroke:#165a71; stroke-width:1; fill:#165a71;opacity:0.5';
$svgArr[] = dotSegArrow([5,1], [3,-1], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
</pre>
<p>This is the more subtle segment at bottom right:</p>
<pre>
$dotAttrs = 'radius:3; stroke:#165a71; stroke-width:2; fill:#165a71';
$segAttrs = 'stroke:#165a71; stroke-width:1';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:#165a71';
$svgArr[] = dotSegArrow([5,-2.5], [2,-3], $dotAttrs, $segAttrs, $arrAttrs,$brdParams);
</pre>

<h3>Drawing Inequalities Graphs</h3>
<p>The above segments can be used to draw inequalities, as follows.</p>

<h4>No <i>y</i>-axis, no grids, no <i>y</i>-axis variable name</h4>
<p>We can "turn off" the <i>y</i>-axis like this:</p>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 40;
$xMin=-3; $xMax=5; $yMin=-1.5;  $yMax=0.5; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$xAxisVbl = "";
$yAxisVbl = "";
$arrows = 0;
$gridAttrs = '';
$svgArr[] = axes(1,'',"labels",0,0, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
doOutput($svgArr,$htmlArr);
?>

<p>Here is the syntax used:</p>
<pre>
$xAxisTicks = 1;	$yAxisTicks = '';
$axisNumbering = "labels";
$verticalGridGap = 0;	$horizontalGridGap = 0;
$gridAttrs = '';
$axesAttrs = 'stroke:#444; stroke-width:1';
$xAxisVbl = "x";	$yAxisVbl = "";
$arrows = 0;
</pre>


<p>The above configuration is useful for presenting inequality solutions, as follows.</p>

 <p>Open one end, closed the other:</p>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 60;
$xMin=-3; $xMax=6; $yMin=-1;  $yMax=1; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$yAxisVbl = "";
$arrows = 0;   // 0|1
// In "axes" function, $dy = 0 means no y-axis
$svgArr[] = axes(1,0,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs,$brdParams

$p = [-2,0]; $q =  [4,0];
$dot0Attrs= 'radius:5; stroke:#165a71; fill:none; stroke-width:1.5';
$segAttrs = 'stroke:#165a71; stroke-width:3';
$dot1Attrs = 'radius:5; stroke:#165a71; fill:#165a71; stroke-width:1.5';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>The above example uses the "dot-segment-dot" type, like this:</p>
<pre>
$p = [-2,0]; $q =  [4,0];
$dot0Attrs= 'radius:5; stroke:#165a71; fill:none; stroke-width:1.5';
$segAttrs = 'stroke:#165a71; stroke-width:3';
$dot1Attrs = 'radius:5; stroke:#165a71; fill:#165a71; stroke-width:1.5';
$svgArr[] = dotSegDot($p, $q, $dot0Attrs,$segAttrs,$dot1Attrs, $brdParams);
</pre>

 <p>Open dot with pointing arrow:</p>
 
 
 <?php

$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 60;
$xMin=-3; $xMax=4; $yMin=-1;  $yMax=1; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$arrows = 0;   // 0|1
$svgArr[] = axes(1,0,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs,$brdParams
//
$p = [2,0]; $q =  [-2,0];
$dot0Attrs= 'radius:5; stroke:#165a71; fill:none; stroke-width:1.5';
$segAttrs = 'stroke:#165a71; stroke-width:3';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:#165a71';
$svgArr[] = dotSegArrow($p, $q, $dot0Attrs,$segAttrs,$arrAttrs, $brdParams);
doOutput($svgArr,$htmlArr);
?>

<p>The line segment, opening circle and final arrow in the above graph are achieved with:
<pre>$p = [2,0]; $q =  [-2,0];
$dot0Attrs= 'radius:5; stroke:#165a71; fill:none; stroke-width:1.5';
$segAttrs = 'stroke:#165a71; stroke-width:3';
$arrAttrs = 'triWidth:10; triHeight:15; stroke:#165a71; stroke-width:1; fill:#165a71';
$svgArr[] = dotSegArrow($p, $q, $dot0Attrs,$segAttrs,$arrAttrs, $brdParams);</pre>

<p>NOTE: <b>dots</b> are always circles, whether the axes are equal scaled or not.</p>



<h2 id="log">Log-log and Semi-log Graphs</h2>
<p>For log-log and semi-logarithmic graphs, we need to call the <code>logAxes</code> function, instead of the ordinary <code>axes</code> function.</p>

<h3>Log-log graphs</h3>
 <div class="imgSvg">
<?php
$boardID = "svgphp-$brdNum";
//if(file_exists("svg/svgphp-$brdNum.svg") && (filemtime("svg/svgphp-$brdNum.svg") > filemtime($_SERVER['DOCUMENT_ROOT']."/".$serverSelf)) && (filemtime("svg/svgphp-$brdNum.svg") > filemtime("../includes/$latestsvgPHPGrapher"))  ) {
	//echo file_get_contents("svg/svgphp-$brdNum.svg");
	//echo "GOT!" ;
//} else {
	$width = 310; $height = 250;
	$xMin=0.1; $xMax=100; $yMin=1;  $yMax=100; 
	$padding = 28;
	$axesType = "loglog"; // log vertical, log horizontal
	list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);

	//
	$gridAttrs = 'stroke:#ccc; stroke-width:1; shape-rendering:crispEdges';
	$axesAttrs = 'stroke:#444; stroke-width:1; shape-rendering:crispEdges';
	$xAxisVbl = "T";	$yAxisVbl = "P";
	$arrows = 1;
	// $dy = '' or 0 means no y-axis
	$svgArr[] = logaxes(1,1,"labels",1,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams

	$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
	$svgArr[] = plot("40+30sin(x/10)",'',$xMax,300, $attrs,$brdParams);
	

	//
	doOutput($svgArr,$htmlArr);
//}
$brdNum++;
?>
</div>


<p>Here is the syntax for the log-log graph type:</p>
<pre>$xMin=0.1; $xMax=100; $yMin=1;  $yMax=100; 
$padding = 28;
$axesType = "loglog"; // log vertical, log horizontal
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$gridAttrs = 'stroke:#ccc; stroke-width:1; shape-rendering:crispEdges';
$axesAttrs = 'stroke:#444; stroke-width:1; shape-rendering:crispEdges';
$xAxisVbl = "T";	$yAxisVbl = "P";
$arrows = 1;
$svgArr[] = logaxes(1,1,"labels",1,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("40+30sin(x/5)",'',$xMax,300, $attrs,$brdParams);
</pre>

<h3>Log-linear graphs</h3>
<p>This is the syntax for the vertical logarithm and horizontal linear graph type:</p>
 <div class="imgSvg">
<?php
$boardID = "svgphp-$brdNum";
if(file_exists("svg/svgphp-$brdNum.svg") && (filemtime("svg/svgphp-$brdNum.svg") > filemtime($_SERVER['DOCUMENT_ROOT']."/".$serverSelf)) && (filemtime("svg/svgphp-$brdNum.svg") > filemtime("../includes/$latestsvgPHPGrapher"))  ) {
	echo file_get_contents("svg/svgphp-$brdNum.svg");
	//echo "GOT!" ;
} else {
	$width = 310; $height = 300;
	$xMin=-4; $xMax=10; $yMin=1;  $yMax=100; 
	$padding = 20;
	$axesType = "loglin"; // log vertical, linear horizontal
	list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);

	//
	$gridAttrs = 'stroke:#ccc; stroke-width:1; shape-rendering:crispEdges';
	$axesAttrs = 'stroke:#444; stroke-width:1; shape-rendering:crispEdges';
	$xAxisVbl = "T";	$yAxisVbl = "P";
	$arrows = 1;
	// $dy = '' or 0 means no y-axis
	$svgArr[] = logaxes(1,1,"labels",1,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams

	$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
	$svgArr[] = plot("40+30sin(x)",'',$xMax,300, $attrs,$brdParams);
	

	//
	doOutput($svgArr,$htmlArr);
}
$brdNum++;
?>
</div>

<p>Here's what you need to do for the log-linear graph type.</p>

<pre>$xMin=-4; $xMax=10; $yMin=1;  $yMax=100; 
$padding = 20;
$axesType = "loglin"; // log vertical, linear horizontal
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);

$gridAttrs = 'stroke:#ccc; stroke-width:1; shape-rendering:crispEdges';
$axesAttrs = 'type:log-linear; stroke:#444; stroke-width:1; shape-rendering:crispEdges';
$xAxisVbl = "T";	$yAxisVbl = "P";
$arrows = 1;

$svgArr[] = logaxes(1,1,"labels",1,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = plot("40+30sin(x)",'',$xMax,300, $attrs,$brdParams);
</pre>


<h2 id="polar">Polar Graphs</h2>
<p>For polar graphs, we need to call the <code>polarAxes</code> function, instead of the ordinary <code>axes</code> function.</p>
<p>Here is the syntax:</p>
<pre>
$rGap = 1;
$degRad = "deg";
$qGap = 15;
$angLabelGap = 30;
$gridAttrs = 'stroke:#ccc; stroke-width:1; fill:none; shape-rendering:crispEdges';
$axesAttrs = 'stroke:#ccc; stroke-width:1; fill:none;';
$svgArr[] = polarAxes($rGap,"labels",$degRad,$qGap,$angLabelGap,  $gridAttrs, $axesAttrs, $brdParams);
</pre>
<p>Here's the graph:</p>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 310;
$xMin=-4; $xMax=4; $yMin=-4;  $yMax='';
$padding = 30; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$rGap = 1;
$degRad = "deg";
$qGap = 15;
$angLabelGap = 30;
$gridAttrs = 'stroke:#ccc; stroke-width:1; fill:none; shape-rendering:crispEdges';
$axesAttrs = 'stroke:#ccc; stroke-width:1; fill:none;';
$svgArr[] = polarAxes($rGap,"labels",$degRad,$qGap,$angLabelGap,  $gridAttrs, $axesAttrs, $brdParams);
//
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = polarplot("3sin(2q)",0,2*pi(),100, $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>The gap between each angular axis (circle) in the grid (and the gap between numbers along the axis):</p>
<pre>$rGap = 1;</pre>
<p>Whether you want radians or degrees for your angle measure:</p>
<pre>$degRad = "deg";</pre>
<p>The gap between the radial axes (the lines) as an angle (in this case 15&deg;):</p>
<pre>$qGap = 15;</pre>
<p>The gap between angle labels (the numbers around the outer circle, in this case 30&deg;):</p>
<pre>$angLabelGap = 30;</pre>
<p>The next part is the standard grid and axes attributes:</p>
<pre>$gridAttrs = 'stroke:#ccc; stroke-width:1; fill:none; shape-rendering:crispEdges';
$axesAttrs = 'stroke:#ccc; stroke-width:1; fill:none;';
$svgArr[] = polarAxes($rGap,"labels",$degRad,$qGap,$angLabelGap,  $gridAttrs, $axesAttrs, $brdParams);
</pre>
<p>Now for the polar curve. Notice we use <code>polarPlot</code> function.</p>
<p>We use "q" for the angle variable (q is commonly used for &theta;).</p>
<pre>$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
$svgArr[] = polarplot("3sin(2q)",0,2*pi(),100, $attrs,$brdParams);</pre>

<p>The function paramaters are:</p>
<pre>$fun,$x_min,$x_max,$points, $attrs,$brdParams</pre>


<h2 id="parametric">Parametric Curve</h2>

<p>Here's a curve traced out by functions in the form `x = x(t)`, and `y=y(t)`. We need to empty the $plist array first, since it may contain values from an example above.</p>
<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 200; $height = 205;
$xMin=-9.6; $xMax=9.6; $yMin=-11.2;  $yMax=''; 
$padding=0;
$axesType = "linlin";
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
//$svgArr[] = axes(10,10,"labels",0,0, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:#5a9fb5;fill-opacity:0.1';
$plist = [];
for ($t = -7*pi()/2; $t < 7*pi()/2+0.08; $t += 0.08) {
	$plist[] = [$t*cos($t), $t*sin($t)];  //1-cos(1.01*$t) - 0.2*cos(8*$t)
}
$svgArr[] = path($plist,'','', $attrs,$brdParams);
doOutput($svgArr,$htmlArr);
?>

<p>Here's the syntax used:</p>

<pre>$plist = [];
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none';
for ($t = -7*pi()/2; $t &lt; 7*pi()/2+0.08; $t += 0.08) {
	$plist[] = [$t*cos($t), $t*sin($t)];
}
$svgArr[] = path($plist,'','', $attrs,$brdParams);</pre>




<h2 id="card">Cardinal Splines</h2>
<p>A <b>cardinal spline</b> is a smooth curve that passes through a set of points. This is different from a regression curve which is an "average" curve, and doesn't necessarily pass through any of the points.</p>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$svgTitle="Cardinal spline";
$width = 310; $height = 200;
$xMin=-4; $xMax=6; $yMin=-3;  $yMax=''; 
$padding=15;
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $svgTitle, $res2, $res3, $padding);
//



$axesAttrs = 'stroke:#444; stroke-width:1; shape-rendering:crispEdges';
$arrows = 1;
$xAxisVbl = "x";	$yAxisVbl = "y";
// $dy = '' or 0 means no y-axis
$gridAttrs = 'stroke:#ccc; stroke-width:1; shape-rendering:crispEdges';
$svgArr[] = axes(1,1,"labels",2,2, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$ptsArr = [ [2,2], [5,-2], [1,-1], [-3,-2], [-3,1] ];
$tension = 0.7;
$isClosed = true;
$numSegs = 20;
$plist = getCurvePoints($ptsArr, $tension, $isClosed, $numSegs);
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:#5a9fb5;fill-opacity:0.1';
$svgArr[] = path($plist,'','', $attrs,$brdParams);
$attrs = 'stroke:none; fill:#165a71';
foreach($ptsArr as $pt) {
	$svgArr[] = dot($pt,3,$attrs, $brdParams);
}
doOutput($svgArr,$htmlArr);
?>

<p>You need to provide a list of points and some attributes, then generate the <code>$plist</code> array. Finally, provide the <code>$plist</code> array to the <code>path</code> function.</p>
<p>The <code>$numSegs</code> changes how smooth the curve is (its the number of points between each supplied point).</p>
<p>Here's the syntax:</p>
<pre>$ptsArr = [ [2,2], [5,-2], [1,-1], [-3,-2], [-3,1] ];
$tension = 0.7;
$isClosed = true;
$numSegs = 20;
$plist = getCurvePoints($ptsArr, $tension, $isClosed, $numSegs);
$attrs = 'stroke:#165a71; stroke-width:1.5; fill:#165a71;fill-opacity:0.15';
$svgArr[] = path($plist,'','', $attrs,$brdParams);
$attrs = 'stroke:none; fill:#165a71';
foreach($ptsArr as $pt) {
  $svgArr[] = dot($pt,3,$attrs, $brdParams);
}</pre>




<h2 id="geom">Geometric Shapes</h2>
<h3>Ellipses and Circles</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-4; $xMax=5; $yMin=-7;  $yMax=7; 
$padding = 10;
//
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";
$arrows = 1;


// $dy = '' or 0 means no y-axis
$svgArr[] = axes(2,2,"labels",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; opacity:0.5';
$svgArr[] = ellipse([-2,-3],1,4,"myEllipse", $attrs, $brdParams); //ellipse($center,$rx,$ry,$id, $attrs, $brdParams)

$attrs = 'stroke:#165a71; stroke-width:3; fill:#ff8; opacity:0.5';
$svgArr[] = circle([3,3],2,"myCirc", $attrs, $brdParams); //circle($center,$radius,$id, $attrs, $brdParams) 

//

doOutput($svgArr,$htmlArr);
?>

<p>Here we've drawn both an ellipse and a circle, using this syntax:</p>
<p class="longCode"><code>$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; opacity:0.5';</code><br />
<code>$svgArr[] = ellipse([-2,-3],1,4,"myEllipse", $attrs, $brdParams); //ellipse($center,$rx,$ry,$id, $attrs, $brdParams)</code></p>
<br />
<pre>$attrs = 'stroke:#165a71; stroke-width:3; fill:#ff8; opacity:0.5';
$svgArr[] = circle([3,3],2,"myCirc", $attrs, $brdParams);
</pre>

<p>However, the circle isn't right because we haven't used equal-scaled axes. Second try:</p>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-4; $xMax=12; $yMin=-7;  $yMax=7; 
$padding = 10;
//
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, '', $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";



// $dy = '' or 0 means no y-axis
$svgArr[] = axes(2,2,"labels",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; opacity:0.5';
$svgArr[] = ellipse([-2,-3],1,4,"myEllipse", $attrs, $brdParams); //ellipse($center,$rx,$ry,$id, $attrs, $brdParams)

$attrs = 'stroke:#165a71; stroke-width:3; fill:#ff8; opacity:0.5';
$svgArr[] = circle([3,3],2,"myCirc", $attrs, $brdParams); //circle($center,$radius,$id, $attrs, $brdParams) 

//

doOutput($svgArr,$htmlArr);
?>

<p>This time, I set <code>$yMax='';</code> so the axes would have equal scaling.</p>


<h3>Arcs</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-4; $xMax=5; $yMin=-3;  $yMax=3; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, '', $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";



// $dy = '' or 0 means no y-axis
$svgArr[] = axes(2,2,"labels",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; opacity:0.5';
$svgArr[] = arc([-2,2],[3,-2],4, $attrs, $brdParams); //arc($start,$end,$radius,$id, $attrs, $brdParams)

$attrs = 'stroke:#165a71; stroke-width:3; fill:none; opacity:0.5';
$svgArr[] = arc([2,0],[0,2],2, $attrs, $brdParams);

//

doOutput($svgArr,$htmlArr);
?>
<p>For an arc, we need to set the beginning and end points, then the radius of the arc. We can also have our arc filled, or not.</p>
<p>This is the first one, which is filled:</p>
<pre>
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; opacity:0.5';
$svgArr[] = arc([-2,2],[3,-2],4, $attrs, $brdParams);</pre>

<p>Now for the second one, without fill:</p>
<pre>$attrs = 'stroke:#165a71; stroke-width:3; fill:none; opacity:0.5';
$svgArr[] = arc([2,0],[0,2],2, $attrs, $brdParams);</pre>

<h3>Angle arcs</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 310;
$xMin=-2; $xMax=2; $yMin=-2;  $yMax=''; 
$padding = 18;

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);

//echo $brdParams['xunitlength'];

// $dy = '' or 0 means no y-axis
$xAxisVbl = "x";$yAxisVbl = "y";
$svgArr[] = axes(1,1,"numbers",0.5, 0.5, $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams

$segAttrs = 'stroke:#165a71; stroke-width:2;';
$arrAttrs = 'triWidth:8; triHeight:12; stroke:#165a71; stroke-width:1; fill:#165a71;';
$arcAttrs = 'stroke:#165a71; stroke-width:1.5; fill:none; opacity:1';
//$p = [0,0];  $q = [1,0];  
//$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);


$attrs = 'stroke:#165a71; stroke-width:1.5; fill:none; opacity:1';

/**/
$p = [0,0];  $q = [1.99,1.99]; 
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
//echo "blip: ".( 180*atan( $q[1]/$q[0])/PI() )."<br>";


$arrAttrs = 'triWidth:6; triHeight:8; stroke:#165a71; stroke-width:1; fill:#165a71;';
$svgArr[] = angleArcArrow([0,0], 40, 0, 180*atan( $q[1]/$q[0])/PI(), $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [0,1.99]; 
$svgArr[] = angleArcArrow([0,0], 50, 0, 90, $arcAttrs, $arrAttrs,$brdParams);;

$p = [0,0];  $q = [-1,2];
$arrAttrs = 'triWidth:8; triHeight:12; stroke:#165a71; stroke-width:1; fill:#165a71;'; 
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
$arrAttrs = 'triWidth:6; triHeight:8; stroke:#165a71; stroke-width:1; fill:#165a71;';
$svgArr[] = angleArcArrow([0,0], 60, 0, 180 + 180*atan( $q[1]/$q[0])/PI(), $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [-1,0];
 
$svgArr[] = angleArcArrow([0,0], 70, 0, 180, $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [-1.5,-2];
$arrAttrs = 'triWidth:8; triHeight:12; stroke:#165a71; stroke-width:1; fill:#165a71;'; 
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
$arrAttrs = 'triWidth:6; triHeight:8; stroke:#165a71; stroke-width:1; fill:#165a71;';
$svgArr[] = angleArcArrow([0,0], 80, 0, 180 + 180*atan( $q[1]/$q[0])/PI(), $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [0,-1]; 
$svgArr[] = angleArcArrow([0,0], 90, 0, 270, $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [1.99,-1]; 
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
$svgArr[] = angleArcArrow([0,0], 100, 0, 360-abs(180*atan( $q[1]/$q[0])/PI()), $arcAttrs, $arrAttrs,$brdParams);

$p = [0,0];  $q = [1,0]; 
$svgArr[] = angleArcArrow([0,0], 110, 0, 359.9, $arcAttrs, $arrAttrs,$brdParams);

//
doOutput($svgArr,$htmlArr);
?>
<p>These are simply an arc with an arrow at the end, using the angleArcArrow function.</p>
<p>Function parameters:</p>
<pre>angleArcArrow($p, $radius, $startAngle, $endAngle, $arcAttrs, $arrAttrs,$brdParams); {  // pixels and degrees</pre>
<p>This is the third angle arc shown (the one in the second quadrant:</p>
<pre>$p = [0,0];  $q = [-1,2]; 
$svgArr[] = segArrow($p, $q,  $segAttrs, $arrAttrs,$brdParams);
$svgArr[] = angleArcArrow([0,0], 60, 0, 180 + 180*atan( $q[1]/$q[0])/PI(), $attrs, $brdParams);</pre>

<h3>Rectangles and Squares</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-4; $xMax=5; $yMin=-3;  $yMax=3; 

list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
$xAxisVbl = "x";	$yAxisVbl = "y";

// $dy = '' or 0 means no y-axis
$svgArr[] = axes(2,2,"labels",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$attrs = 'stroke:#165a71; stroke-width:1; fill:#5a9fb5; fill-opacity:0.1';
$svgArr[] = rect([-3.5,-1],[-1,3], $attrs,$brdParams); //rect($p,$q,$attrs,$brdParams)

$attrs = 'stroke:#165a71; fill:#ff8; stroke-width:1.5; rx:8; opacity:0.7';
$svgArr[] = square([1, -2], 2.5, $attrs,$brdParams);  //square($p,$squareWidth, $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>Here's the syntax for the rectangle. The points are the bottom left, then top right of the rectangle.</p>
<pre>
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; fill-opacity:0.1';
$svgArr[] = rect([-3.5,-1],[-1,3], $attrs,$brdParams);
</pre>
<p>Now for the square. We specify the bottom-left of the square, and the next parameter is its width.</p>
<pre>
$attrs = 'stroke:#165a71; fill:#ff8; stroke-width:1.5; rx:8; opacity:0.7';
$svgArr[] = square([1, -2], 2.5, $attrs,$brdParams);
</pre>

<p>Rounded corners are achieved using  <code>rx:8</code></p>



<h3>Triangles</h3>

<?php
$boardID = "svgphp-$brdNum"; $brdNum++;
$width = 310; $height = 230;
$xMin=-4; $xMax=5; $yMin=-3;  $yMax=3; 
list($svgArr, $htmlArr, $brdParams) = initBoard($boardID, $width, $height, $xMin, $xMax, $yMin, $yMax, $axesType, $res1, $res2, $res3, $padding);
//
$xAxisVbl = "x";	$yAxisVbl = "y";
$svgArr[] = axes(2,2,"labels",'','', $xAxisVbl, $yAxisVbl, $gridAttrs, $axesAttrs, $arrows, $brdParams);  //$dx,$dy,$labels,$gdx,$gdy, $xAxisVbl, $yAxisVbl, $attrs,$brdParams
//
$p = [-4,-2]; $q =  [-1,2]; $r = [-2,-2];
$attrs = 'stroke:#165a71; stroke-width:1; fill:#5a9fb5; fill-opacity:0.1';
$svgArr[] = triangle($p,$q,$r, $attrs,$brdParams);
$p = [-1,0]; $q =  [4,0]; $r = [2,1];
$attrs = 'stroke:#165a71; fill:#ff8; stroke-width:1.5; opacity:0.7';
$svgArr[] = triangle($p,$q,$r, $attrs,$brdParams);
//
doOutput($svgArr,$htmlArr);
?>

<p>For triangles, we specify the 3 vertices. Here's the first one:</p>
<pre>$p = [-4,-2]; $q =  [-1,2]; $r = [-2,-2];
$attrs = 'stroke:#165a71; stroke-width:3; fill:#5a9fb5; fill-opacity:0.1';
$svgArr[] = triangle($p,$q,$r, $attrs,$brdParams);</pre>

<p>Now for the second one:</p>
<pre>

$p = [-1,0]; $q =  [4,0]; $r = [2,1];
$attrs = 'stroke:#165a71; fill:#ff8; stroke-width:1.5; opacity:0.7';
$svgArr[] = triangle($p,$q,$r, $attrs,$brdParams);</pre>





<script src="/includes/cssjs/<?php echo $latestASCIIMathTeXKaTeX; ?>"></script>
<script src="/includes/cssjs/katex/<?php echo $latestKatexJs; ?>"></script> 
<script>
AMfunc(true);
</script>
<?php

include('../includes/nonLoopFooter.php');
?>